unit ListList_OOP_D7U;
{$OPTIMIZATION ON}
{ Hierarchische Listenstrukturen am Beispiel (simulierter) Dateien und Ordner.
  SingleEntry = einzelne Datei, Basisklasse
  ListEntry = Verzeichnis, fhrt eine Liste mit Single- und ListEntry-Elementen

  Die Struktur wird per Zufallsgenerator aufgebaut, wobei die Wahrscheinlichkeit
  fr das Erzeugen neuer ListEntry-Elemente mit jeder zustzlichen Hierarchie-
  Ebene sinkt:
    if (Random(10*Level) < 1) then <New List, Rekursion mit Level+1>
     else <New Single Entry>
  Das Element der obersten Ebene ist eine Liste, entspricht dem Stammverzeichnis.

  Prozedurale Variante untersucht das Laufzeitverhalten von
  - Objektverwaltung (1 Mio Objekte)
  - Stringmanipulationen
  - Listen bzw. Arrays (20 Millionen Suchvorgnge)
  - Random-Generator
  - Stackverwaltung

  Objektorientierte Variante verkapselt Hinzufgen und Suchen vollstndig.
  Die in den Methoden ausgefhrten Operationen sind dieselben wie in der
  prozeduralen Version
}
interface

uses
  Windows, Messages, SysUtils, Variants, Classes, Graphics, Controls, Forms,
  Dialogs, StdCtrls, StrUtils, MMSystem;

type
  TSingleEntry = class;

  TSingleEntry = class  // einzelnes Element
    private
      FEntryName: String;
    protected
      ParentList: TSingleEntry;  // backtracking fr full name
    public
      constructor Create(EName: String);
      property EntryName: String read FEntryName;
      function GetDisplayName: String; virtual;
      function FindEntry(PartialName: String): TSingleEntry; virtual;
      function AddEntry(EName: String): TSingleEntry; virtual;
      function AddNode(NName: String): TSingleEntry; virtual;
      function FirstEntry: TSingleEntry; virtual;
      function NextEntry(CurEntry: TSingleEntry): TSingleEntry; virtual;
      function FullName: String;
    end;

  TListEntry = class(TSingleEntry)   // Verzeichnis
    private
      EntryList: TList; // TSingleEntry und TListEntry gemischt
    public
      constructor Create(EName: String);
      destructor Destroy; override;
      function GetDisplayName: String; override;
      function FindEntry(PartialName: String): TSingleEntry; override;
      function AddEntry(EName: String): TSingleEntry; override;
      function AddNode(NName: String): TSingleEntry; override;
      function FirstEntry: TSingleEntry; override;
      function NextEntry(CurEntry: TSingleEntry): TSingleEntry; override;

    end;

type
  TLLForm = class(TForm)
    bCreateTree: TButton;  // Demo: Struktur mit 100 Elementen
    bBenchCreate: TButton; // 10000 Elemente, 100 mal aufgebaut
    ListBox1: TListBox;
    procedure bCreateTreeClick(Sender: TObject);
    procedure bBenchCreateClick(Sender: TObject);
  private
    procedure BuildBaseList(ECount: Integer);
    procedure ShowEntry(S: String);
    procedure ClearList;
  public
    EntryCount: Integer; // Runterzhler frs Anlegen der Struktur
    RootDir: TListEntry; // "Stammverzeichnis"
    procedure BuildListList(List: TSingleEntry; Level: Integer);
    function FindListEntry(List: TSingleEntry; SubStr: String): String;
    procedure PrintListList(List: TSingleEntry; Level: Integer);
  end;

var
  LLForm: TLLForm;

implementation
{$R *.dfm}


// Demo-Liste mit 100 Elementen, Anzeige
procedure TLLForm.bCreateTreeClick(Sender: TObject);
begin
  ClearList;
  ListBox1.Clear;
  BuildBaseList(100);
  PrintListList(RootDir, 1);
  ClearList;
end;

procedure TLLForm.ClearList;
begin
  if RootDir <> nil then RootDir.Free;
  RootDir := TListEntry.Create('C:');
end;

// Legt die Elementenzahl fest und erzeugt das Stammverzeichnis
procedure TLLForm.BuildBaseList(ECount: Integer);
begin
  EntryCount := ECount;
  while EntryCount > 0 do
    BuildListList(RootDir,1);
end;

// Listenaufbau (rekursiv)
procedure TLLForm.BuildListList(List: TSingleEntry; Level: Integer);
var x, y, LocalCount, NameLength: Integer;
    NewName: String;
begin
  // maximal 25 Eintrge pro Liste - nur das Stammverzeichnis
  // kann mehr haben
  LocalCount := Random(25)+1;
  for x := 1 to LocalCount do
  begin
    // zuflliger Name mit 4-11 Zeichen
    NameLength := Random(8)+1;
    NewName := '';
    for y := 1 to NameLength do
      NewName := NewName + Chr(Ord('a')+Random(26));

    // Einzelner Eintrag oder neue Liste? Die Wahrscheinlichkeit
    // fr neue Listen (und weitere Rekursion) sinkt mit zunehmender
    // Verschachelungstiefe
    if Random(10*Level) < 1
      then BuildListList(List.AddNode(NewName), Level+1)
       else List.AddEntry(NewName);

    Dec(EntryCount);
    if EntryCount <= 0 then Break;
  end;
end;

// Anzeige in der Listbox, auch fr Suchergebnisse
procedure TLLForm.ShowEntry(S: String);
begin
  ListBox1.Items.Add(S);
end;

// Ausgabe der Demo-Liste (100 Elemente)
procedure TLLForm.PrintListList(List: TSingleEntry; Level: Integer);
var x: Integer; Lead: String;
  CurEntry: TSingleEntry;
begin
  Lead := '';
  for x := 1 to Level-1 do Lead := Lead + '  ';
  // EntryName+' (LIST) fr Verzeichnisse
  ShowEntry(Lead+List.GetDisplayName);
  // Dateieintrge und Verzeichnisse, ungeordnet
  CurEntry := List.FirstEntry;
  while CurEntry <> nil do
  begin
    PrintListList(CurEntry, Level+1);
    CurEntry := List.NextEntry(CurEntry);
  end;
end;

function TLLForm.FindListEntry(List: TSingleEntry; SubStr: String): String;
var E: TSingleEntry;
begin
  E := List.FindEntry(SubStr);
  if E <> nil then Result := E.FullName;
end;

// 1 Million Objekte, 24 Millionen Suchvorgnge
procedure TLLForm.bBenchCreateClick(Sender: TObject);
var x, y: Integer;
    ConstructionTime, DestructionTime, FindTime: TDateTime;
    FullBench, Start, Meantime: TDateTime;

  function TStr(Cap: String;Time: TDateTime): String;
  var Temp: String;
  begin
    Temp := LongTimeFormat;
    LongTimeFormat := 'nn:ss:zzz';
    Result := Format('%-25s'#9'%s'#13#10, [Cap, TimeToStr(Time)]);
    LongTimeFormat := Temp;
  end;

begin
  ListBox1.Clear;
  ClearList;
  ConstructionTime := 0; DestructionTime := 0; FindTime := 0;
  Start := Now;
  for x := 1 to 10 do
  begin
    Meantime := Now;
    BuildBaseList(100000);
    ConstructionTime := ConstructionTime + (Now-Meantime);
    Meantime := Now;
    ShowEntry('Search a: ' + FindListEntry(RootDir, 'a'));
    ShowEntry('Search ax: ' + FindListEntry(RootDir, 'ax'));
    ShowEntry('Search axv: ' + FindListEntry(RootDir, 'axv'));
    ShowEntry('Search axve: ' + FindListEntry(RootDir, 'axve'));
    for y := 1 to 20 do
      FindListEntry(RootDir,'X'); // gibts nicht
    FindTime := FindTime + (Now-MeanTime);
    ListBox1.Update;
    MeanTime := Now;
    ClearList;  // in Delphi direkter Destruktoraufruf
    DestructionTime := DestructionTime+ (Now-Meantime);
  end;
  FullBench := Now-Start;

  ShowMessage(TStr('Time:',        FullBench)+
              TStr('Construction:',constructionTime)+
              TStr('Destruction:' ,destructionTime)+
              TStr('Find:',        findTime));

end;

// ===========================================

{ TSingleEntry }
constructor TSingleEntry.Create(EName: String);
begin
  FEntryName := EName;
end;

function TSingleEntry.AddEntry(EName: String): TSingleEntry;
begin
  raise Exception.Create('TSingleEntry.AddEntry: not implemented');
end;

function TSingleEntry.AddNode(NName: String): TSingleEntry;
begin
  raise Exception.Create('TSingleEntry.AddNode: not implemented');
end;


function TSingleEntry.FindEntry(PartialName: String): TSingleEntry;
begin
  if PartialName = EntryName
    then Result := Self
    else Result := nil;
end;

function TSingleEntry.GetDisplayName: String;
begin
  Result := EntryName;
end;

function TSingleEntry.FirstEntry: TSingleEntry;
begin
  Result := nil;
end;

function TSingleEntry.NextEntry(CurEntry: TSingleEntry): TSingleEntry;
begin
  Result := nil;
end;

function TSingleEntry.FullName: String;
begin
  if ParentList <> nil
    then Result := ParentList.FullName+'\'+EntryName
    else Result := EntryName;
end;

{ TListEntry }
constructor TListEntry.Create(EName: String);
begin
  EntryList := TList.Create;
  inherited;
end;

// In Delphi ist ein Destruktor ntig, der die Listen durchluft
destructor TListEntry.Destroy;
var x: Integer; E: TSingleEntry;
begin
  with EntryList do
    for x := 0 to Count-1 do
    begin
      E := Items[x];
      E.Free;
    end;
  EntryList.Free;
end;

function TListEntry.AddEntry(EName: String): TSingleEntry;
begin
  Result := TSingleEntry.Create(EName);
  EntryList.Add(Result);
  Result.ParentList := Self;
end;

function TListEntry.AddNode(NName: String): TSingleEntry;
begin
  Result := TListEntry.Create(NName);
  EntryList.Add(Result);
  Result.ParentList := Self;
end;

function TListEntry.FindEntry(PartialName: String): TSingleEntry;
var x: Integer; E: TSingleEntry;
begin
  Result := inherited FindEntry(PartialName);
  if Result = nil then
    for x := 0 to EntryList.Count-1 do
    begin
      E := EntryList[x];
      Result := E.FindEntry(PartialName);
      if Result <> nil then break;
    end;
end;


function TListEntry.GetDisplayName: String;
begin
  Result := inherited GetDisplayName + ' (LIST)';
end;

function TListEntry.FirstEntry: TSingleEntry;
begin
  if EntryList.Count = 0 then Result := nil
   else Result := EntryList[0];
end;

function TListEntry.NextEntry(CurEntry: TSingleEntry): TSingleEntry;
var x: Integer;
begin
  x := EntryList.IndexOf(CurEntry);
  if (x <> -1) and (x+1 < EntryList.Count)
    then Result := EntryList[x+1]
    else Result := nil;
end;

begin
  Randomize;
end.
